#include <vga.h>
#include <stdio.h>
#include <unistd.h>
#include "keys.h"
#include "bd.h"

#include <sys/time.h>
#include <sys/stat.h>
#include <dirent.h>
#include <errno.h>

struct timeval tv1,tv2,tv3;
struct timezone tz;

par_t *first_par, *last_par, *clipboard;

int ystart;

static char cursor_buf[4096];
static char cursor_buf2[4096];
static int pcx, pcy, pcline;
static int bltop, blbot;
int lasty;

int cursor_x, cursor_y, cursor_height;

pos_t sel_st, sel_end;
int pos;
pos_t sf,sl;

int WIDTH;
int page_width;
int page_height;
int page_right_margin;
int page_left_margin;
int page_top_margin;
int page_bottom_margin;

int autosavekeys;
int autosavekeyinterval;

int hebmode;

int curfont;

int qfonts[10];

int display_par(par_t *, int, int);

int clear_flags() {
    par_t *par;
    for(par=first_par;par!=NULL;par=par->next){ 
        par->cacheflags=0;
    }
    return 0;
}


int force_redraw(par_t *par, int line) {
    if(par==NULL) {
        par_t *tmp;
        tmp=first_par;
        while(tmp!=NULL) {
            force_redraw(tmp,-1);
            tmp=tmp->next;
        }
    } else if(line==-1) {
        int i;
        for(i=0;i<par->numlines;i++)par->lines[i]->flags|=1;
    } else par->lines[line]->flags|=1;
    return 0;
}

int clear_clipboard(void) {
    par_t *par, *next;
    
    par=clipboard;
    while(par!=NULL) {
        next=par->next;
        free_par(par);
        par=next;
    }
    clipboard=NULL;
    return 0;
}

par_t *copy_range(pos_t from, pos_t to) {
    int s;
    par_t *src, *dest, *dr;
    
    src=from.par;
    s=from.pos;
    dest=NULL;
    while(src!=to.par) {
        if(dest==NULL) {
            dr=dup_par(src,s,strlen(src->text));
            dr->prev=NULL;
            dest=dr;
        } else {
            dr->next=dup_par(src,s,strlen(src->text));
            dr->next->prev=dr;
            dr=dr->next;
        }
        src=src->next;
        s=0;
    }
    
    if(src!=NULL) {
        if(dest==NULL) {
            dr=dup_par(src,s,to.pos);
            dr->prev=NULL;
            dest=dr;
        } else {
            dr->next=dup_par(src,s,to.pos);
            dr->next->prev=dr;
            dr=dr->next;
        }
    }
    
    dr->next=NULL;
    return dest;
    

}

pos_t paste(pos_t pos, par_t *text) {
    par_t *p1, *p2;
    pos_t from,to;
    
    split_par(pos.par,pos.pos);
    p1=pos.par->next;
    from.par=text;
    from.pos=0;
    to.par=NULL;
    p2=copy_range(from,to);
    p2->prev=pos.par;
    pos.par->next=p2;
    while(p2->next!=NULL)p2=p2->next;
    p1->prev=p2;
    p2->next=p1;
    if(pos.par->next==p2)p2=pos.par;
    join_par(pos.par);
    join_par(p2);
    to.par=p2;
    to.pos=strlen(p2->text);
    return to;
}

int selection(void) {
    if((sel_st.par!=NULL)&&pos_prior(sel_st,sel_end))return 1; else return 0;
}

int selected(par_t *par, int pos) {
    pos_t p={pos,par};
    
    if(!selection())return 0; 
    if(!pos_prior(p,sel_st)&&pos_prior(p,sel_end)) return 1;
    return 0;
}

int modify_select(pos_t f, pos_t l) {
    if(!selection()) {
        if(pos_prior(f,l)) {
            sel_st=f;
            sel_end=l;
        } else {
            sel_st=l;
            sel_end=f;
        }
    } else { /* many cases */
        if(pos_prior(f,l)) {
            if(pos_equal(f,sel_st)) {
                if(pos_prior(l,sel_end)) {
                    sel_st=l;
                } else {
                    sel_st=sel_end;
                    sel_end=l;
                }
            } else if(pos_equal(f,sel_end)) {
                sel_end=l;
            } else {
                sel_st=f;
                sel_end=l;
            }
        } else {
            if(pos_equal(f,sel_end)) {
                if(pos_prior(sel_st,l)) {
                    sel_end=l;
                } else {
                    sel_end=sel_st;
                    sel_st=l;
                }
            } else if(pos_equal(f,sel_st)) {
                sel_st=l;
            } else {
                sel_st=l;
                sel_end=f;
            }
        }

    }
    return 0;
}

int fontof(par_t *par, int pos) {
    int j=par->numfontchanges-1;
    
    while((par->fontchangepos[j]>pos)&&(j>0))j--;
    
    return j;
}   

void display_curfont(void) {
    char str[30];
    int x;
    
    blank_area(left_skip-left_spare+1,0,left_skip-left_spare+1+8*ui_unit,ui_unit);
    ft_font=ui_sfont->chars;
    sprintf(str,"%i",(int)fonts[curfont&0xfff]->height>>6);
    ft_writen(left_skip-left_spare+1,ui_unit*2/3,3,str);
    if(fonts[curfont&0xfff]->face->style_flags&2) {
        drawchar(ui_font->chars,'B',left_skip-left_spare+ui_unit,ui_unit*2/3,color_decor,color_bg,1);
    }
    if(fonts[curfont&0xfff]->face->style_flags&1) {
        drawchar(ui_font->chars,'I',left_skip-left_spare+3*ui_unit/2,ui_unit*2/3,color_decor,color_bg,1);
    }
    x=ft_writen(left_skip-left_spare+2*ui_unit,ui_unit*2/3,20,fonts[curfont&0xfff]->face->family_name);
    if(curfont&0x10000) {
        drawline(left_skip-left_spare+2*ui_unit,ui_unit*2/3-ui_sfont->ul_pos,x,ui_unit*2/3-ui_sfont->ul_pos,ft_fg);
    }
}

void drawvertruler(int ystart) {
    int y;
    int i, p;
    
    blank_area(left_skip-left_spare-25,top_skip+1,24*screen_bpp,vis_pix_height);
    p=ystart/page_height;
    y=ystart-p*page_height;
    i=y/72;
    i++;
    y=i*72+p*page_height-ystart;
    if(y<12) {
        y+=72;
        i++;
    }
    drawchar(ui_font->chars,'0'+(p+1)%10,left_skip-left_spare-25,top_skip+20,color_decor,color_bg,0);
    if(p>10)
        drawchar(ui_sfont->chars,'0'+(p+1)/10,left_skip-left_spare-35,top_skip+20,color_decor,0,1);
    while (y<vis_height-20) {
        if(i*72>=page_height) {
            i=0;
            p++;
            y=p*page_height-ystart;
            drawchar(ui_font->chars,'0'+(p+1)%10,left_skip-left_spare-25,top_skip+y*DPI/72,color_decor,color_bg,0);
            if(p>10)
                drawchar(ui_sfont->chars,'0'+(p+1)/10,left_skip-left_spare-35,top_skip+(y*DPI/72),color_decor,0,1);
        }
        drawline(left_skip-left_spare-4,top_skip+(y*DPI/72),left_skip-left_spare-1,top_skip+(y*DPI/72),color_decor);
        drawchar(ui_sfont->chars,'0'+i%10,left_skip-left_spare-11,top_skip+(y*DPI/72),color_decor,0,1);
        if(i>9)
            drawchar(ui_sfont->chars,'0'+i/10,left_skip-left_spare-17,top_skip+(y*DPI/72),color_decor,0,1);
        i++;
        y+=72;
    }
}

void drawhorzruler(par_t *par) {
    int x, i;
    blank_area(left_skip-left_spare,top_skip-28,vis_pitch,27);

    for(i=0; i*72<=WIDTH;i++) {
        x=(WIDTH-i*72)*DPI/72+left_skip;
        drawline(x,top_skip-14,x,top_skip-12,color_decor);  
        drawchar(ui_sfont->chars,'0'+i%10,x-2,top_skip-16,color_decor,0,1);
    }

    drawline(left_skip,top_skip-12,left_skip+WIDTH*DPI/72,top_skip-11,color_decor);
    if(par!=NULL) {
        x=(WIDTH-par->par_point_margin)*DPI/72+left_skip;
        drawline(x,top_skip-13,x,top_skip-13,color_decor);  
        drawline(x-1,top_skip-14,x+1,top_skip-14,color_decor);  
        drawline(x-2,top_skip-15,x+2,top_skip-15,color_decor);  
        x=(WIDTH-par->par_point_margin-par->par_point_width)*DPI/72+left_skip;
        drawline(x,top_skip-13,x,top_skip-13,color_decor);  
        drawline(x-1,top_skip-14,x+1,top_skip-14,color_decor);  
        drawline(x-2,top_skip-15,x+2,top_skip-15,color_decor);  
        x=(WIDTH-par->par_point_margin-par->firstlineindent)*DPI/72+left_skip;
        drawline(x,top_skip-11,x,top_skip-11,color_decor);  
        drawline(x-1,top_skip-10,x+1,top_skip-10,color_decor);  
        drawline(x-2,top_skip-9,x+2,top_skip-9,color_decor);  
        for(i=0;i<par->numtabs;i++) if(par->tabs[i]<par->par_point_width) {
            x=(WIDTH-par->par_point_margin-par->tabs[i])*DPI/72+left_skip;
            drawline(x,top_skip-10,x,top_skip-7,color_decor);  
            drawline(x-2,top_skip-7,x+3,top_skip-7,color_decor);  
        }
        drawline(left_skip-left_spare+3,top_skip-27,left_skip-9,top_skip-27,color_decor); 
        drawline(left_skip-left_spare+3,top_skip-7,left_skip-9,top_skip-7,color_decor); 
        drawline(left_skip-9,top_skip-27,left_skip-9,top_skip-7,color_decor); 
        drawline(left_skip-left_spare+3,top_skip-27,left_skip-left_spare+3,top_skip-7,color_decor); 
        switch((par->style&0x0c)>>2) {
            case 0:
                i=left_spare/2-8;
                x=left_skip-11-i;
                break;
            case 1:
                i=left_spare/2-8;
                x=left_skip-left_spare+5;
                break;
            case 2:
                i=left_spare/2-8;
                x=left_skip-left_spare+5+i/2;
                break;
            case 3:
                i=left_spare-16;
                x=left_skip-left_spare+5;
                break;
        }
        drawline(x,top_skip-23,x+i,top_skip-23,color_decor);
        drawline(x,top_skip-22,x+i,top_skip-22,color_decor);
        drawline(x,top_skip-18,x+i,top_skip-18,color_decor);
        drawline(x,top_skip-17,x+i,top_skip-17,color_decor);
        drawline(x,top_skip-13,x+i,top_skip-13,color_decor);
        drawline(x,top_skip-12,x+i,top_skip-12,color_decor);
    }
}

int changesize( int x, int y, int dpi) {
    int olddpi, i;
    
    olddpi=DPI;

    if(x!=display_width || y!=display_pix_height) {
        display_width=x;
        display_pix_height=y;
        DPI=dpi;
        set_size();
        change_fontres(ui_font,UDPI);
        change_fontres(ui_sfont,UDPI);
        graphics_changesize();
        ui_unit=ui_font->pix_height>>6;
        ui_x=UDPI;
        ui_y=UDPI;
        ui_height=10*ui_unit;
        ui_width=15*ui_unit;
    } else {
        DPI=dpi;
        set_size();
    }

    if(DPI!=olddpi) {
        for(i=0;i<fontsnum;i++)change_fontres(fonts[i],DPI);
    }
    
    blank_area(0,0,screen_pitch,display_pix_height);
    drawline(0,top_skip-1,display_width-1,top_skip-1,color_decor);
    drawline(left_skip-left_spare-1,0,left_skip-left_spare-1,display_pix_height-1,color_decor);
    
    clear_flags();
    
    reformat(1,DPI,first_par);
    display_par(first_par,-1,0x1301);
    
    drawhorzruler(NULL);
    drawvertruler(0);

    return 0;
}

int writen(int x, int y, int n, unsigned char *s, par_t *par, int line)
{
    font_element *ft_font;
    int i=0;
    int j, k;
    int ox, sx;
    int lu, lux;
    FT_font *font;
    
    lu=0;
    lux=-1;
    x<<=6;
    ox=x;
    sx=0;
    
    ft_font=par->font->chars;
    while((i<n) && s[i] && ((x>>6)+ft_font[s[i]].bitmap.width)*screen_bpp<screen_pitch) {
        j=0;
        k=par->v2l[i+par->lines[line]->st]+par->lines[line]->st;
        while((j<par->numfontchanges-1)&&(k>=par->fontchangepos[j+1]))j++;
        
        font=fonts[par->fontchange[j]&0xfff];
        if((lu==0)&&(par->fontchange[j]&0x10000)) {
            lu=1;
            lux=x;
        } else if ((lu==1)&&!(par->fontchange[j]&0x10000)) {
            int i;
            for(i=y-par->font->ul_pos;i<y-par->font->ul_pos+par->font->ul_thickness;i++)
                drawline(lux>>6,i,x>>6,i,ft_fg);
            lu=0;
        }
        if(selected(par,k)) {
            int q;
            int ascender=(font->ascender>>6)-1;
            int descender=font->descender>>6;
            if(sx==0)sx=x>>6;
            for(q=y-ascender; q<y-descender; q++)
                drawline(sx,q,(x+font->chars[par->text[k]].metrics.horiAdvance)>>6,q,ft_fg);
            if(s[i]<32) {
                sx=x>>6;
                i++;
            } else {
                sx=(x+font->chars[par->text[k]].metrics.horiAdvance)>>6;
                x+=drawchar(font->chars,s[i++],x>>6,y,ft_bg,ft_fg,1);
            }
        } else {
            sx=0;
            if(s[i]<32) {
                i++;
            } else {
                x+=drawchar(font->chars,s[i++],x>>6,y,ft_fg,ft_bg,ft_transparent);
            }
        }
    }

    if(lu==1){
        int i;
        for(i=y-par->font->ul_pos;i<y-par->font->ul_pos+par->font->ul_thickness;i++)
            drawline(lux>>6,i,x>>6,i,ft_fg);
    }    
    return x-ox;
}

int writeline_r2l(int xoffs, int y, par_t *par, int line, int dpi)
{
    int i=0;
    int dir=1;
    int j, k, x, p, lu, lux;
    int sx;
    FT_font *font;

    if(par->lines[line]->len==0)return 0;
    
    lu=0;
    lux=-1;
    sx=0;

    while(i<par->lines[line]->len) {
        j=0;
        if(dir==0) {
            p=i+par->lines[line]->st;
        } else {
            p=par->lines[line]->visend-i;
        }
        k=par->v2l[p]+par->lines[line]->st;
        
        if(i==0)
            x=xoffs-(par->xpos[k]>>6);

        font=fonts[par->cfont[k]&0xfff];

        if((lu==0)&&(par->cfont[k]&0x10000)) {
            lu=1;
            lux=x;
        } else if ((lu==1)&&!(par->cfont[k]&0x10000)) {
            int i;
            for(i=y-par->font->ul_pos;i<y-par->font->ul_pos+par->font->ul_thickness;i++)
                drawline(x,i,lux,i,ft_fg);
            lu=0;
        }
        
        if(par->text[k]>31)
            x=xoffs-((par->xpos[k]+font->chars[par->text[k]].metrics.horiAdvance)>>6);
        else {
            int k;
            if (i<par->lines[line]->len-1) {
                int p;
                if(dir==0) {
                    p=i+1+par->lines[line]->st;
                } else {
                    p=par->lines[line]->visend-i-1;
                }
                k=par->v2l[p]+par->lines[line]->st;
            } else k=par->lines[line]->visend+1;
            x=xoffs-(par->xpos[k]>>6);
        }
        if(selected(par,k)) {
            int q;
            int ascender=(font->ascender>>6)-1;
            int descender=font->descender>>6;
            if(sx==0)
                sx=xoffs-(par->xpos[k]>>6);
            for(q=y-ascender; q<y-descender; q++)
                drawline(x,q,sx,q,ft_fg);
            sx=x;
            if(par->text[k]>31)
                drawchar(font->chars,par->text[k],x,y,ft_bg,ft_fg,1);
        } else {
            sx=0;
            if(par->text[k]>31)
                drawchar(font->chars,par->text[k],x,y,ft_fg,ft_bg,ft_transparent);
        }
        i++;
    }
    if(lu==1){
        int i;
        for(i=y-par->font->ul_pos;i<y-par->font->ul_pos+par->font->ul_thickness;i++)
            drawline(x,i,lux,i,ft_fg);
    }    
    return 0;
}

int findx(par_t *par, int line, int fx)
{
    int i=0;
    int dir=1;
    int j, k, x, p, lk, lx;
    int xoffs;
    
    xoffs=calc_xoffs_r2l(par,line,DPI,&j,&j);
    x=xoffs;
    if(fx>=x) { 
        return par->lines[line]->st;
    }
    while((i<=par->lines[line]->len)&&(x>fx)) {
        j=0;
        if(dir==0) {
            p=i+par->lines[line]->st;
        } else {
            p=par->lines[line]->visend-i;
        }
        lk=k;
        lx=x;
        if(i<par->lines[line]->len)
            k=par->v2l[p]+par->lines[line]->st; else k=par->lines[line]->visend+1;
        x=xoffs-((par->xpos[k])>>6);
        i++;
    }
    if(fx-x < lx-fx)lk=k;
    if(lk>strlen(par->text))lk=strlen(par->text);
    return lk;
}

int calc_xoffs_r2l(par_t *par, int i, int dpi, int *multiplier, int *divider) {
    int xoffs;
    int j;
    
    *multiplier=1;
    switch((par->style&0x1c)>>2) {
        case 0:
            xoffs=DISPWIDTH-par->par_point_margin*dpi/72;
            break;
        case 1:
            xoffs=DISPWIDTH-par->par_point_margin*dpi/72
                +par->lines[i]->width+par->lines[i]->indent
                -par->par_point_width*dpi/72;
            break;
        case 2:
            xoffs=DISPWIDTH-(par->par_point_margin*dpi/72)-
                  (-par->lines[i]->width-par->lines[i]->indent+(par->par_point_width*dpi/72))/2;
            break;
        case 3:
            j=0;
            xoffs=DISPWIDTH-par->par_point_margin*dpi/72;
            break;
        case 4:
            xoffs=(par->par_point_margin+par->par_point_width)*dpi/72-par->lines[i]->width-
                            par->lines[i]->indent;
            break;
        case 5:
            xoffs=par->par_point_margin*dpi/72;
            break;
        case 6:
            xoffs=(par->par_point_margin*2+par->par_point_width)*dpi/144-par->lines[i]->width/2;
            break;
        case 7: /* should be fully justified. fix later */
        default:
            xoffs=0;
    }
    
    xoffs+=left_skip;
    
    return xoffs;
}

int display_par_simple(par_t *par, int bdline, int bdx, int bdcd, int pos, int flags, int dpi, int ystart) {
/* 
flags:
    0x001 - draw
    0x010 - don't blank 
    0x100 - draw all par
    0x200 - don't draw cursor
*/

    int yoffs, i;
    int x, cy, xoffs;
    int ascender;

    par->lastline=bdline;

    yoffs=par->par_point_ypos-ystart;    
    ascender=(par->font->ascender>>6)-1;

    for(i=0;i<par->numlines;i++) {
        int y;
        int multiplier;
        int divider;

        y=(yoffs+par->lines[i]->yoffs)*dpi/72+top_skip;    
        if((y>=top_skip)&&(y+par->lines[i]->height*dpi/72<=display_pix_height)) {
            if(((flags&1)==1)||((i==bdline)&&(flags&0x200)==0)){
    		xoffs=calc_xoffs_r2l(par, i, dpi, &multiplier, &divider);
            }

            if(par->lines[i]->flags&1) {
                par->lines[i]->flags&=~1;
                if((flags&1)==1) {
                    if((flags&0x10)==0) {
                        int b;
                    
                        b=y;
                        blbot=y+par->line_pix_height;
                        if(blbot>display_pix_height)blbot=display_pix_height;
                        blank_area(left_skip-left_spare,bltop,vis_pitch,blbot-bltop);
                        bltop=blbot;
                        } else bltop=y+par->line_pix_height;
                    lasty=y+par->line_pix_height+1;
                    writeline_r2l(xoffs,y+ascender,par,i,dpi);
                } else bltop=y+par->line_pix_height;
            };
    
            if ((i==bdline)&&(flags&0x200)==0) {
    
                bdx=par->xpos[pos];
                if(multiplier!=1) {
                    bdx=bdx*multiplier/divider;
                }
                
                pcline=bdline;
                pcy=par->par_point_ypos+par->lines[i]->yoffs+1;
                x=xoffs-(bdx>>6);
                pcx=x;
                cy=y+par->line_pix_height-1;
    
                cursor_height=par->line_pix_height;
                cursor_y=y;
                cursor_x=x;
                if ((bdcd&1)==1) cursor_x-=3;
                save_area(cursor_x,cursor_y,4*screen_bpp,cursor_height,cursor_buf);
    
                drawline(x,y,x,cy,color_cursor);
                if ((bdcd&1)==1) {
                    drawline(x-1,cy-4,x-1,cy-1,color_cursor);
                    drawline(x-2,cy-3,x-2,cy-2,color_cursor);
                } else {
                    drawline(x+1,cy-4,x+1,cy-1,color_cursor);
                    drawline(x+2,cy-3,x+2,cy-2,color_cursor);
                }
            }
        }
    }
    return 0;
}

par_t *findy(int y, par_t *par, int *line, int dir) {
    int i;

    if(y>last_par->par_point_ypos+last_par->par_point_height) {
        *line=last_par->numlines-1;
        return last_par;
    }

    while((par!=NULL)&&(par->par_point_ypos+par->par_point_height < y)) {
        par=par->next;
    }
    
    i=0;

    if(par==NULL) {
        par=last_par;
    } 

    while((i<par->numlines)&&((par->par_point_ypos+par->lines[i]->yoffs) <= y)) i++;

    if(dir<0) {
        *line=i-1;
        if (*line==-1)*line=0;
    } else {
        if(i==par->numlines) {
            if(par->next==NULL) {
                *line=i-1;
            } else {
                *line=0;
                par=par->next;
            }
        } else *line=i;
    }
    
    return par;
}
#if 1
int display_par(par_t *par, int pos, int flags)
/* flags:
    bit 0 - draw the paragraph
    bit 1 - only cursor moved
    bit 8 - redraw all paragraph
    bit 9 - hide cursor
    bit 10 - redraw from this paragraph on
    bit 11 - font was changed
    bit 12 - redraw
   return:
    -2: paragraph too low to display

*/
{
    par_t *tmp;
    int i, yoffs, f;
    int bdx, bdline, bdcd, bdpos;

    reformat(0, DPI, first_par);
    force_redraw(NULL,0);
    
    yoffs=par->par_point_ypos-ystart;    

    par_calc_cursor(par, DPI, pos, &bdx, &bdline, &bdcd, &bdpos);

    if(((flags&0x200)==0)&&(((yoffs+par->lines[bdline]->yoffs)<0) || 
       ((yoffs+par->lines[bdline]->yoffs+par->lines[bdline]->height)>=vis_height))) {
        int diff;
        par_t *tmp;
        int i;
        
        flags &=~2;
        flags |= 1;
        if(yoffs+par->lines[bdline]->yoffs<0) {
            diff=yoffs+par->lines[bdline]->yoffs;
        } else {
            diff=yoffs+par->lines[bdline]->yoffs+par->lines[bdline]->height-vis_height;
        }

        ystart+=diff;
       
    	tmp=findy(ystart,first_par,&i,diff);
        ystart=tmp->par_point_ypos+tmp->lines[i]->yoffs;
        drawvertruler(ystart);
        
        yoffs=(par->par_point_ypos-ystart);    
    }

    tmp=findy(ystart,first_par,&i,1);
    tmp=first_par;
    bltop=top_skip+1;

    if(flags&2) f=0x10; else f=0x01;

    while((i!=-2)&&(tmp!=NULL)) {
        if(tmp==par) {
            i=display_par_simple(par, bdline, bdx, bdcd, pos, f, DPI, ystart);
        } else i=display_par_simple(tmp,0,0,0,0,0x201,DPI,ystart);
        tmp=tmp->next;
    }
    if((flags&1)&&(bltop<display_pix_height)) { 
        blank_area(left_skip-left_spare,bltop,vis_pitch,display_pix_height-bltop);
        return -2;
    }
    return 0;
}
#else
int display_par(par_t *par, int pos, int flags)
/* flags:
    bit 0 - draw the paragraph
    bit 8 - redraw all paragraph
    bit 9 - hide cursor
    bit 10 - redraw from this paragraph on
    bit 11 - font was changed
    bit 12 - redraw
   return:
    -2: paragraph too low to display

*/
{
    int bdx, bdline, bdcd, bdpos;
    int yoffs;
    int i;
    
    yoffs=par->par_point_ypos-ystart;    

    if((flags&0x200)&&(par->par_point_ypos-ystart>vis_height)) return -2;
    if((flags&0x200)&&(yoffs+par->par_point_height<0)) return -3;

    break_paragraph(par, strlen(par->text), DPI);
    par_calc_cursor(par, DPI, pos, &bdx, &bdline, &bdcd, &bdpos);
    reformat(0, DPI, par);

    if(((flags&0x200)==0)&&(((yoffs+par->lines[bdline]->yoffs)<0) || 
       ((yoffs+par->lines[bdline]->yoffs+par->lines[bdline]->height)>=vis_height))) {
        int diff;
        par_t *tmp;
        int i;
        
        if(yoffs+par->lines[bdline]->yoffs<0) {
            diff=yoffs+par->lines[bdline]->yoffs;
        } else {
            diff=yoffs+par->lines[bdline]->yoffs+par->lines[bdline]->height-vis_height;
        }

        ystart+=diff;
       
    	tmp=findy(ystart,first_par,&i,diff);
        ystart=tmp->par_point_ypos+tmp->lines[i]->yoffs;
        drawvertruler(ystart);
        
        blank_area(left_skip-left_spare,top_skip,vis_pitch,vis_pix_height);
        i=0;
        while((i!=-2)&&(tmp!=NULL)) {
            if(tmp!=par)
                i=display_par_simple(tmp,0,0,0,0,0x211,DPI,ystart);
            tmp=tmp->next;
        }
  
        yoffs=(par->par_point_ypos-ystart);    
        flags&=~0x400;
        flags|=0x101;
    }

    if((flags&0x1400)||(i==1)) {
        par_t *tmp=par->next;
        int f;
        
        f=0x201;
        if(flags&0x1000) {
            tmp=first_par;
            f|=0x10;
            blank_area(left_skip-left_spare,top_skip,vis_pitch,vis_pix_height);
        }
        i=0;
        reformat(0,DPI,par);
        while((i!=-2)&&(tmp!=NULL)) {
            if(tmp!=par)
                i=display_par_simple(tmp,0,0,0,0,f,DPI,ystart);
            tmp=tmp->next;
        }
        if(last_par->par_point_height+last_par->par_point_ypos-ystart<vis_height) {
            int y;
            y=(last_par->par_point_height+last_par->par_point_ypos-ystart)*DPI/72+top_skip;
            blank_area(left_skip-left_spare,y,vis_pitch,vis_pix_height-y);
        }
    }

    display_par_simple(par, bdline, bdx, bdcd, pos, flags, DPI, ystart);

    return 0;
}
#endif

int edit_par(par_t *par, int hpos)
{
    int ch;
    int refresh;
    int f;
    int ret;
    pos_t oldpos;
    
    if(par!=curpos.par) return K_F10;
    pos=curpos.pos;
    if(par==NULL)return K_F10;
    
    f=257;

    ret=0;

    switch(hpos) {
        case -2:
            f|=0x1400;
            break;
        case -3:
            f|=0x800;
            break;
        default:
            break;
    }

    drawhorzruler(par);
    display_par(par,pos,f);
    curfont=par->fontchange[fontof(par,pos)];
    display_curfont();
    
    while (1) {    
    
        autosavekeys++;
        if(autosavekeys >= autosavekeyinterval ) {
            autosavekeys=0;
            save(autosavefilename, first_par);
        }
        ch=readchar();
        if(cursor_height)save_area(cursor_x,cursor_y,4*screen_bpp,cursor_height,cursor_buf2);
        if(cursor_height)restore_area(cursor_x,cursor_y,4*screen_bpp,cursor_height,cursor_buf);

        refresh=0;
        
        if((ch>=' ')&&(ch<127)) {
            if(hebmode)ch=hebconv(ch);
            insert_char(par, ch, pos);
            refresh=1;
    	} else {
            int i;
            
            switch(ch) {

/* special chars */
                case SHIFT+ K_ENTER:
                    insert_char(par, 31, pos);
                    refresh=1;
                    break;
                case K_TAB:
                    insert_char(par, 9, pos);
                    refresh=1;
                    break;
                case K_ENTER: 
                    split_par(par,pos);
                    refresh=0x601;
                    curpos=pos_next(curpos);
                    break;

/* deletion */
                case SHIFT + K_DEL:
                    if(selection()) {
                        del_range(sel_st,sel_end);
                        curpos=sel_st;
                        sel_st.par=NULL;
                        refresh=1;
                    }
                    break;
                case K_DEL:
                    del_char(par,pos);
                    refresh=1;
                    break;
                case K_BACKSPACE:
                    curpos=pos_prev(curpos);
                    if(pos!=curpos.pos || par!=curpos.par)del_char(curpos.par,curpos.pos);
                    refresh=1;
                    break;

/* cursor movement */
                case K_RIGHT:
                    curpos=pos_prev(curpos);
                    refresh=0x01000002;
                    break;
                case K_LEFT:
                    curpos=pos_next(curpos);
                    refresh=0x01000002;
                    break;
                case K_UP:
                    if(pcline>0) {
                        curpos.pos=findx(par,pcline-1,pcx);
                    }else if(par->prev!=NULL) {
                        curpos.par=par->prev;
                        curpos.pos=findx(curpos.par,curpos.par->numlines-1,pcx);
                    }
                    refresh=0x01000002;
                    break;
                case K_DOWN:
                    if(pcline<par->numlines-1) {
                        curpos.pos=findx(par,pcline+1,pcx);
                    } else if(par->next!=NULL) {
                        curpos.par=par->next;
                        curpos.pos=findx(curpos.par,0,pcx);
                    }
                    refresh=0x01000002;
                    break;
                case K_PGUP:
                    curpos.par=findy(pcy-vis_height,first_par,&i,1);
                    curpos.pos=findx(curpos.par,i,pcx);
                    refresh=0x01000002;
                    break;
                case K_PGDOWN:
                    curpos.par=findy(pcy+vis_height,first_par,&i,-1);
                    curpos.pos=findx(curpos.par,i,pcx);
                    refresh=0x01000002;
                    break;
                case CONTROL + K_PGUP:
                    curpos.par=first_par;
                    curpos.pos=0;
                    break;
                case CONTROL + K_PGDOWN:
                    curpos.par=last_par;
                    curpos.pos=strlen(curpos.par->text);
                    break;
                case CONTROL + K_HOME:
                    curpos.pos=0;
                    refresh=0x01000002;
                    break;
                case CONTROL + K_END:
                    curpos.pos=strlen(par->text);
                    refresh=0x01000002;
                    break;
                case K_HOME: 
                    curpos.pos=findx(par,pcline,9999);
                    refresh=0x01000002;
                    break;
                case K_END: 
                    curpos.pos=findx(par,pcline,0);
                    refresh=0x01000002;
                    break;

/* selection */
                case SHIFT + K_UP:
                    SETPOS(sf,pos,par);
                    if(pcline>0) {
                        curpos.pos=findx(par,pcline-1,pcx);
                        SETPOS(sl,curpos.pos,par);
                        modify_select(sf,sl);
                        refresh=0x1101;
                    }else if(par->prev!=NULL) {
                        curpos.par=par->prev;
                        curpos.pos=findx(curpos.par,curpos.par->numlines-1,pcx);
                        SETPOS(sl,curpos.pos,curpos.par);
                        modify_select(sf,sl);
                        refresh=0x1101;
                    }
                    break;
                case SHIFT + K_DOWN:
                    SETPOS(sf,pos,par);
                    if(pcline<par->numlines-1) {
                        curpos.pos=findx(par,pcline+1,pcx);
                        SETPOS(sl,curpos.pos,par);
                        modify_select(sf,sl);
                        refresh=0x1101;
                    } else if(par->next!=NULL) {
                        curpos.par=par->next;
                        curpos.pos=findx(curpos.par,0,pcx);
                        SETPOS(sl,curpos.pos,curpos.par);
                        modify_select(sf,sl);
                        refresh=0x1101;
                    }
                    break;
                case SHIFT + K_RIGHT:
                    if((pos!=0) || (par!=first_par)) {
                        SETPOS(sf,pos,par);
                        curpos=sl=pos_prev(sf);
                        modify_select(sf,sl);
                        refresh=0x1101;
                    }
                    break;
                case SHIFT + K_LEFT:
                    if((pos!=strlen(par->text)) || (par!=last_par)) {
                        SETPOS(sf,pos,par);
                        curpos=sl=pos_next(sf);
                        modify_select(sf,sl);
                        refresh=0x1101;
                    }
                    break;
                case SHIFT + K_PGDOWN:
                    SETPOS(sf,pos,par);
                    curpos.par=findy(pcy+vis_height,first_par,&i,-1);
                    curpos.pos=findx(curpos.par,i,pcx);
                    SETPOS(sl,curpos.pos,curpos.par);
                    modify_select(sf,sl);
                    refresh=0x1101;
                    break;
                case SHIFT + K_PGUP:
                    SETPOS(sf,pos,par);
                    curpos.par=findy(pcy-vis_height,first_par,&i,1);
                    curpos.pos=findx(curpos.par,i,pcx);
                    SETPOS(sl,curpos.pos,curpos.par);
                    modify_select(sf,sl);
                    refresh=0x1101;
                    break;
                case CONTROL + 'k': /* cancel selection */
                    sel_st.par=NULL;
                    refresh=0x1101;
                    break;

/* clipboard */
    		case CONTROL + 'v':
                    if(clipboard!=NULL) {
                        oldpos=curpos;
                        curpos=paste(curpos,clipboard);
                        if(!volatile_selection) {
                            sel_st=oldpos;
                            sel_end=curpos;
                        }
                        refresh=0x1001101;
                        ret=-2;
                    }
                    break;
                case ALT + 'c': 
                    if(clipboard==NULL)clear_clipboard();
                    if(selection())clipboard=copy_range(sel_st,sel_end);
                    break;
                    
/* paragraph style */
                case CONTROL+'r':
                    set_par_style(par,~0x0c,0,0);
                    refresh=0x2000101;
                    break;
                case CONTROL+'l':
                    set_par_style(par,~0x0c,4,0);
                    refresh=0x2000101;
                    break;
                case CONTROL+'z':
                    set_par_style(par,~0x0c,8,0);
                    refresh=0x2000101;
                    break;
                case CONTROL+'j':
                    set_par_style(par,~0x0c,12,0);
                    refresh=0x2000101;
                    break;
                case CONTROL+'e':
                    set_par_style(par,~0,0,3);
                    refresh=0x2000101;
                    break;
                case K_F4:
                    choose_parameters(par);
                    refresh=0x2001101;
                    break;
                case K_F5:
                    choose_tabs(par);
                    refresh=0x2001101;
                    break;

                case CONTROL+'h':
                    hebmode=!hebmode;
                    break;
                case CONTROL + '0':
                case CONTROL + '1':
                case CONTROL + '2':
                case CONTROL + '3':
                case CONTROL + '4':
                case CONTROL + '5':
                case CONTROL + '6':
                case CONTROL + '7':
                case CONTROL + '8':
                case CONTROL + '9':
                    i=ch - CONTROL -'0';
                    if((qfonts[i]>=0)&&((qfonts[i]&0xfff)<fontsnum)) {
                        curfont=qfonts[i];
                        display_curfont();
                        if (selection()) {
                            set_font_range(sel_st,sel_end,curfont);
                            refresh=0x1101;
                        }
                    }
                    break;
                case K_F1: 
                    if((i=choose_font(fonts[curfont&0xfff]->face->family_name,
                        	      fonts[curfont&0xfff]->height>>6,
                    		      fonts[curfont&0xfff]->face->style_flags,
                                      curfont&0xfffff000))>=0) {
                        if (selection()) {
                            set_font_range(sel_st,sel_end,i);
                            refresh=0x1101;
                        }
    			curfont=i;
                        display_curfont();
                    }
                    break;
                case CONTROL + SHIFT + '0':
                case CONTROL + SHIFT + '1':
                case CONTROL + SHIFT + '2':
                case CONTROL + SHIFT + '3':
                case CONTROL + SHIFT + '4':
                case CONTROL + SHIFT + '5':
                case CONTROL + SHIFT + '6':
                case CONTROL + SHIFT + '7':
                case CONTROL + SHIFT + '8':
                case CONTROL + SHIFT + '9':
                    qfonts[ch - CONTROL - SHIFT - '0']=curfont;
                    break;
                case CONTROL+'p':
                    sel_st.par=NULL;
                    print_ps();
                    refresh=0x01000002;                    
                    break;
                case K_F3: 
                case K_F10: 
                    ret=ch;
                    break;
                case K_F2:
                    if(choose_file_name("Save as:",filename)>=0) {
                        if(strlen(filename)>0)save(filename, first_par);
                        makeautosavename(autosavefilename, filename);
                        autosavekeys=0;
                    }
                    break;

                case CONTROL+'q':
                    refresh=0x1101;
                    break;
                case K_F11:
                    changesize(display_width, display_pix_height, DPI/2);
                    refresh=0x2000002;
                    break;                    
                case SHIFT+K_F11:
                    changesize(display_width, display_pix_height, DPI*2);
                    refresh=0x2000002;
                    break;                    
                case K_F12:
                    printf("%i\n",0/(i-i));
                    break;
            }
        } 
        if(curpos.par!=par){
            if(ret==0)ret=-1;
            refresh &=0xfd000000;
        }
        pos=curpos.pos;
        
        if(refresh&0x1000000){
            curfont=par->fontchange[fontof(par,pos)];
            if(volatile_selection&&selection()) {
                sel_st.par=NULL;
                refresh|=0x1101;
            }
            display_curfont();
        }
        if(refresh&0x2000000){
            drawhorzruler(par);
        }
        
    	if(refresh&0xffffff) { 
            display_par(par,pos,refresh&0xffffff);
        }
        if(ret)return ret;
        if(!(refresh&0xffffff) && cursor_height)
            restore_area(cursor_x,cursor_y,4*screen_bpp,cursor_height,cursor_buf2);
    }

    return 0;
}

int main()
{
    int i;
    int n;
    par_t *par;
    char cfname[256];
    char *c;
    struct stat sb;

    c=getenv("HOME");
    strncpy(cfname,c,240);
    strcat(cfname,"/.timna/");
    if(stat(cfname, &sb)) { 
        if(errno==ENOENT) mkdir(cfname,448);
    } else {
        if(!S_ISDIR(sb.st_mode)) {
            fprintf(stderr, "Error: %s should be a directory.\n",cfname);
        }
    }

    readconfigfile();
    for(i=0;i<10;i++)qfonts[i]=i;
    hebmode=1;
    
    fontsnum=0;
    init_fonts();
    
    init_graphics();

    if(open_ui_font("Arial",0,12<<6,UDPI)) {
        printf("Error openning ui font.\n");
        close_graphics();
        exit(1);
    } 

    if(open_font("Arial",0,12<<6,DPI)) {
        printf("Error openning default font.\n");
        close_graphics();
        exit(1);
    } 

    ui_unit=ui_font->pix_height>>6;
    ui_x=UDPI;
    ui_y=UDPI;
    ui_height=10*ui_unit;
    ui_width=15*ui_unit;

    blank_area(0,0,screen_pitch,display_pix_height);
    drawline(0,top_skip-1,display_width-1,top_skip-1,color_decor);
    drawline(left_skip-left_spare-1,0,left_skip-left_spare-1,display_pix_height-1,color_decor);
    
    drawvertruler(0);
    ft_fg=color_fg;
    ft_bg=color_bg;
    ft_transparent=1;
    
    makeautosavename(autosavefilename, filename);
    autosavekeys=0;
    
    sel_st.par=NULL;

    if(load(template)) {
        first_par=new_par(NULL);
        last_par=first_par;
        save(template,first_par);
    }

    clipboard=NULL;
    par=first_par;
    par->par_point_ypos=0;
    n=0;
    ystart=0;
    curfont=0;
    curpos.par=par;
    curpos.pos=0;
    
    while ((i=edit_par(par,n))!=K_F10) {
        switch(i) {
            case -1:
                par=curpos.par;
                n=-1;
                break;
            case -2:
                par=curpos.par;
                n=-2;
                break;
            case 1:
                par=sel_st.par;
                sel_st.par=NULL;
                n=pos;
                display_par(par,0,0x601);
                break;
            case K_F3:
                while(first_par!=NULL)del_par(first_par);
                if(choose_file_name("Load filename:",filename)>=0) {
                    makeautosavename(autosavefilename, filename);
                    autosavekeys=0;
                    if(load(filename)==-1) {
                        printf("Can't read %s\n",filename);
                    }
                    n=-2;
                    ystart=0;
                    drawvertruler(0);
                    par=first_par;
                    curpos.par=par;
                    curpos.pos=0;
                }
                break;
        }                
    }
    
    close_graphics();
    
    return 0;
}

