/***************************************************************************/
/*                                                                         */
/*  freetype.h                                                             */
/*                                                                         */
/*    FreeType high-level API and common types (specification only).       */
/*                                                                         */
/*  Copyright 1996-1999 by                                                 */
/*  David Turner, Robert Wilhelm, and Werner Lemberg.                      */
/*                                                                         */
/*  This file is part of the FreeType project, and may only be used        */
/*  modified and distributed under the terms of the FreeType project       */
/*  license, LICENSE.TXT.  By continuing to use, modify, or distribute     */
/*  this file you indicate that you have read the license and              */
/*  understand and accept it fully.                                        */
/*                                                                         */
/***************************************************************************/

#ifndef FREETYPE_H
#define FREETYPE_H


#define _FREETYPE_


#define FREETYPE_MAJOR 2
#define FREETYPE_MINOR 0


#ifndef EXPORT_DEF
#define EXPORT_DEF  extern
#endif

#include <fterrors.h>
#include <ftsystem.h>
#include <ftimage.h>

#ifdef __cplusplus
  extern "C" {
#endif


  typedef unsigned char  FT_Bool;


  typedef signed short    FT_FWord;   /* Distance in FUnits */


  typedef unsigned short  FT_UFWord;  /* Unsigned distance */


  typedef signed char  FT_Char;


  typedef unsigned char  FT_Byte;


  typedef char  FT_String;


  typedef signed short  FT_Short;


  typedef unsigned short  FT_UShort;


  typedef int  FT_Int;


  typedef unsigned int  FT_UInt;


  typedef signed long  FT_Long;


  typedef unsigned long  FT_ULong;


  typedef signed short  FT_F2Dot14;


  typedef signed long  FT_F26Dot6;


  typedef signed long  FT_Fixed;


  typedef int  FT_Error;


  typedef struct  FT_UnitVector_
  {
    FT_F2Dot14  x;
    FT_F2Dot14  y;

  } FT_UnitVector;


  typedef struct  FT_Matrix_
  {
    FT_Fixed  xx, xy;
    FT_Fixed  yx, yy;

  } FT_Matrix;


  typedef struct  FT_BBox_
  {
    FT_Pos  xMin, yMin;
    FT_Pos  xMax, yMax;

  } FT_BBox;


#define FT_MAKE_TAG( _x1, _x2, _x3, _x4 ) \
          (((FT_ULong)_x1 << 24) |        \
           ((FT_ULong)_x2 << 16) |        \
           ((FT_ULong)_x3 << 8)  |        \
            (FT_ULong)_x4)


  typedef struct FT_ListNodeRec_*  FT_ListNode;


  typedef struct FT_ListRec_*  FT_List;


  typedef struct  FT_ListNodeRec_
  {
    FT_ListNode  prev;
    FT_ListNode  next;
    void*        data;

  } FT_ListNodeRec;


  typedef struct  FT_ListRec_
  {
    FT_ListNode  head;
    FT_ListNode  tail;

  } FT_ListRec;


#define FT_IS_EMPTY(list)  ( (list).head == 0 )


  typedef struct  FT_Glyph_Metrics_
  {
    FT_Pos  width;         /* glyph width  */
    FT_Pos  height;        /* glyph height */

    FT_Pos  horiBearingX;  /* left side bearing in horizontal layouts */
    FT_Pos  horiBearingY;  /* top side bearing in horizontal layouts  */
    FT_Pos  horiAdvance;   /* advance width for horizontal layout     */

    FT_Pos  vertBearingX;  /* left side bearing in vertical layouts */
    FT_Pos  vertBearingY;  /* top side bearing in vertical layouts  */
    FT_Pos  vertAdvance;   /* advance height for vertical layout    */

  } FT_Glyph_Metrics;


  typedef void  (*FT_Generic_Finalizer)(void*  object);


  typedef struct  FT_Generic_
  {
    void*                 data;
    FT_Generic_Finalizer  finalizer;

  } FT_Generic;


  typedef struct  FT_Bitmap_Size_
  {
    FT_Short  height;
    FT_Short  width;

  } FT_Bitmap_Size;


  typedef struct FT_LibraryRec_  *FT_Library;


  typedef struct FT_DriverRec_*  FT_Driver;


  typedef struct FT_FaceRec_*  FT_Face;


  typedef struct FT_SizeRec_*  FT_Size;


  typedef struct FT_GlyphSlotRec_*  FT_GlyphSlot;


  typedef struct FT_CharMapRec_*  FT_CharMap;


  typedef enum  FT_Encoding_
  {
    ft_encoding_none    = 0,
    ft_encoding_symbol  = 0,
    ft_encoding_unicode = FT_MAKE_TAG('u','n','i','c'),
    ft_encoding_latin_2 = FT_MAKE_TAG('l','a','t','2'),
    ft_encoding_sjis    = FT_MAKE_TAG('s','j','i','s'),
    ft_encoding_big5    = FT_MAKE_TAG('b','i','g','5'),

    ft_encoding_adobe_standard = FT_MAKE_TAG('a','d','o','b'),
    ft_encoding_adobe_expert   = FT_MAKE_TAG('a','d','b','e'),
    ft_encoding_adobe_custom   = FT_MAKE_TAG('a','d','b','c'),

    ft_encoding_apple_roman    = FT_MAKE_TAG('a','r','m','n')


  } FT_Encoding;

  typedef struct  FT_CharMapRec_
  {
    FT_Face      face;
    FT_Encoding  encoding;
    FT_UShort    platform_id;
    FT_UShort    encoding_id;

  } FT_CharMapRec;





  typedef struct  FT_FaceRec_
  {
    FT_Driver        driver;
    FT_Memory        memory;
    FT_Stream        stream;

    FT_Long          num_faces;
    FT_Long          face_index;

    FT_Generic       generic;

    FT_GlyphSlot     glyph;

    FT_Size          size;

    FT_CharMap       charmap;
    
    FT_Int           num_charmaps;
    FT_CharMap*      charmaps;

    FT_ListRec       sizes_list;

    void*            extensions;

    FT_Long          face_flags;
    FT_Long          style_flags;

    FT_Long          num_glyphs;

    FT_String*       family_name;
    FT_String*       style_name;

    FT_Int           num_fixed_sizes;
    FT_Bitmap_Size*  available_sizes;


    FT_BBox          bbox;

    FT_UShort        units_per_EM;
    FT_Short         ascender;
    FT_Short         descender;
    FT_Short         height;

    FT_Short         max_advance_width;
    FT_Short         max_advance_height;

    FT_Short         underline_position;
    FT_Short         underline_thickness;

    FT_UShort        max_points;
    FT_Short         max_contours;

  } FT_FaceRec;


#define FT_FACE_FLAG_SCALABLE  1


#define FT_FACE_FLAG_FIXED_WIDTH  4


#define FT_FACE_FLAG_FIXED_SIZES  2


#define FT_FACE_FLAG_SFNT  8


#define FT_FACE_FLAG_HORIZONTAL  0x10


#define FT_FACE_FLAG_VERTICAL  0x20


#define FT_FACE_FLAG_KERNING  0x40


#define FT_FACE_FLAG_FAST_GLYPHS  0x80


#define FT_HAS_HORIZONTAL(face)  (face->face_flags & FT_FACE_FLAG_HORIZONTAL)
#define FT_HAS_VERTICAL(face)    (face->face_flags & FT_FACE_FLAG_VERTICAL)
#define FT_HAS_KERNING(face)     (face->face_flags & FT_FACE_FLAG_KERNING)
#define FT_IS_SCALABLE(face)     (face->face_flags & FT_FACE_FLAG_SCALABLE)
#define FT_IS_SFNT(face)         (face->face_flags & FT_FACE_FLAG_SFNT)
#define FT_IS_FIXED_WIDTH(face)  (face->face_flags & FT_FACE_FLAG_FIXED_WIDTH)
#define FT_HAS_FIXED_SIZES(face) (face->face_flags & FT_FACE_FLAG_FIXED_SIZES)
#define FT_HAS_FAST_GLYPHS(face) (face->face_flags & FT_FACE_FLAG_FAST_GLYPHS)


#define FT_STYLE_FLAG_ITALIC  1


#define FT_STYLE_FLAG_BOLD  2



  typedef struct  FT_Size_Metrics_
  {
    FT_UShort   x_ppem;        /* horizontal pixels per EM               */
    FT_UShort   y_ppem;        /* vertical pixels per EM                 */

    FT_Fixed    x_scale;       /* two scales used to convert font units  */
    FT_Fixed    y_scale;       /* to 26.6 frac. pixel coordinates..      */

    FT_Pos      ascender;      /* ascender in 26.6 frac. pixels          */
    FT_Pos      descender;     /* descender in 26.6 frac. pixels         */
    FT_Pos      height;        /* text height in 26.6 frac. pixels       */
    FT_Pos      max_advance;   /* max horizontal advance, in 26.6 pixels */

  } FT_Size_Metrics;


  typedef struct  FT_SizeRec_
  {
    FT_Face          face;      /* parent face object              */
    FT_Generic       generic;   /* generic pointer for client uses */
    FT_Size_Metrics  metrics;   /* size metrics                    */

  } FT_SizeRec;



  typedef struct FT_SubGlyph_  FT_SubGlyph;
  
  struct FT_SubGlyph_
  {
    FT_Int        index;
    FT_UShort     flags;
    FT_Int        arg1;
    FT_Int        arg2;
    FT_Matrix     transform;
  };


#define FT_SUBGLYPH_FLAG_ARGS_ARE_WORDS            1
#define FT_SUBGLYPH_FLAG_ARGS_ARE_XY_VALUES        2
#define FT_SUBGLYPH_FLAG_ROUND_XY_TO_GRID          4
#define FT_SUBGLYPH_FLAG_SCALE                     8
#define FT_SUBGLYPH_FLAG_XY_SCALE               0x40
#define FT_SUBGLYPH_FLAG_2X2                    0x80
#define FT_SUBGLYPH_FLAG_USE_MY_METRICS        0x200


  typedef struct  FT_GlyphSlotRec_
  {
    FT_Face           face;
    FT_GlyphSlot      next;

    FT_Glyph_Metrics  metrics;
    FT_Glyph_Metrics  metrics2;

    FT_Glyph_Tag      format;
    FT_Bitmap         bitmap;
    FT_Outline        outline;
    
    FT_Int            num_subglyphs;
    FT_Int            max_subglyphs;
    FT_SubGlyph*      subglyphs;

    void*             control_data;
    void*             control_len;

    void*             other;

  } FT_GlyphSlotRec;


  EXPORT_DEF
  FT_Error  FT_Init_FreeType( FT_Library*  library );


  EXPORT_DEF
  FT_Error  FT_Done_FreeType( FT_Library  library );


  typedef struct FT_Open_Args_
  {
    FT_Byte*     memory_base;
    FT_Long      memory_size;
    FT_String*   pathname;
    FT_Stream    stream;
    FT_Driver    driver;

  } FT_Open_Args;

  EXPORT_DEF
  FT_Error  FT_New_Face( FT_Library   library,
                         const char*  filepathname,
                         FT_Long      face_index,
                         FT_Face*     face );


  EXPORT_DEF
  FT_Error  FT_Open_Face( FT_Library    library,
                          FT_Open_Args* args,
                          FT_Long       face_index,
                          FT_Face*      face );


  EXPORT_DEF
  FT_Error  FT_Attach_File( FT_Face      face,
                            const char*  filepathname );
                            
  EXPORT_DEF
  FT_Error  FT_Attach_Stream( FT_Face       face,
                              FT_Open_Args* parameters );

  EXPORT_DEF
  FT_Error  FT_Done_Face( FT_Face  face );


  EXPORT_DEF
  FT_Error  FT_New_Size( FT_Face   face,
                         FT_Size*  size );

  EXPORT_DEF
  FT_Error  FT_Done_Size( FT_Size  size );

  EXPORT_DEF
  FT_Error  FT_Set_Char_Size( FT_Face     face,
                              FT_F26Dot6  char_width,
                              FT_F26Dot6  char_height,
                              FT_UInt     horz_resolution,
                              FT_UInt     vert_resolution );


  EXPORT_DEF
  FT_Error  FT_Set_Pixel_Sizes( FT_Face    face,
                                FT_UInt    pixel_width,
                                FT_UInt    pixel_height );


  EXPORT_DEF
  FT_Error  FT_New_GlyphSlot( FT_Face        face,
                              FT_GlyphSlot*  aslot );


  EXPORT_DEF
  void  FT_Done_GlyphSlot( FT_GlyphSlot  slot );


  EXPORT_DEF
  FT_Error  FT_Load_Glyph( FT_Face  face,
                           FT_UInt  glyph_index,
                           FT_Int   load_flags );

  EXPORT_DEF
  FT_Error  FT_Load_Char( FT_Face   face,
                          FT_ULong  char_code,
                          FT_Int    load_flags );

#define FT_LOAD_NO_SCALE  1


#define FT_LOAD_NO_HINTING  2


#define FT_LOAD_NO_OUTLINE  4


#define FT_LOAD_NO_BITMAP  8


#define FT_LOAD_LINEAR  16


#define FT_LOAD_PEDANTIC  128


#define FT_LOAD_IGNORE_GLOBAL_ADVANCE_WIDTH 512

#define FT_LOAD_NO_RECURSE 1024

#define FT_LOAD_DEFAULT  0


  EXPORT_DEF
  FT_Error  FT_Get_Glyph_Bitmap( FT_Face     face,
                                 FT_Bitmap*  map );


  EXPORT_DEF
  FT_Error  FT_Get_Kerning( FT_Face     face,
                            FT_UInt     left_glyph,
                            FT_UInt     right_glyph,
                            FT_Vector*  kerning );



#if 0
  EXPORT_DEF
  FT_Error  FT_Select_Charmap( FT_Face      face,
                               FT_Encoding  encoding );


  EXPORT_DEF
  FT_Error  FT_Set_Charmap( FT_Face     face,
                            FT_CharMap  charmap );
#endif

  EXPORT_DEF
  FT_UInt  FT_Get_Char_Index( FT_Face   face,
                              FT_ULong  charcode );


  EXPORT_DEF
  FT_Long  FT_MulDiv( FT_Long  a,
                      FT_Long  b,
                      FT_Long  c );


  EXPORT_DEF
  FT_Long  FT_MulFix( FT_Long  a,
                      FT_Long  b );


  EXPORT_DEF
  FT_Long  FT_DivFix( FT_Long  a,
                      FT_Long  b );


  EXPORT_DEF
  int  FT_Decompose_Outline( FT_Outline*        outline,
                             FT_Outline_Funcs*  interface,
                             void*              user );
 

  EXPORT_DEF
  FT_Error  FT_New_Outline( FT_Library   library,
                            FT_UInt      numPoints,
                            FT_Int       numContours,
                            FT_Outline*  outline );


  EXPORT_DEF
  FT_Error  FT_Done_Outline( FT_Library   library,
                             FT_Outline*  outline );

  EXPORT_DEF
  FT_Error  FT_Get_Outline_CBox( FT_Outline*  outline,
                                 FT_BBox*     cbox );


  EXPORT_DEF
  void  FT_Translate_Outline( FT_Outline*  outline,
                              FT_Pos       xOffset,
                              FT_Pos       yOffset );


  EXPORT_DEF
  FT_Error  FT_Set_Raster( FT_Library            library,
                           FT_Raster_Interface*  interface,
                           FT_Raster             raster );


  EXPORT_DEF
  FT_Error  FT_Set_Raster_Mode( FT_Library    library,
                                FT_Glyph_Tag  format,
                                const char*   mode,
                                const char*   args );


#ifdef __cplusplus
  }
#endif


#endif /* FREETYPE_H */
